#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netdb.h>
#include <time.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>

#include "config.h"

#if (linux)
#define __FAVOR_BSD
#endif

#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>

extern int errno;

#ifndef NOFILE
#define NOFILE 1024
#endif

#define ALARM_TIMEOUT	15 
#define SCAN_TIMEOUT	15

volatile int alarm_flag;
time_t now;

volatile void die(char *s, int n)
{
	fprintf(stderr, s, n);
	exit(1);
}

int do_ignore(unsigned short lport, unsigned short hport)
{
	int i;
	unsigned short p;
	for(i = 0; i < sizeof ignore_ports / sizeof(int) ; i++){
		p = htons(ignore_ports[i]);
		if (lport == p && hport == p) return 1;
	}
	return 0;
}

char *hostlookup(int i)
{
	static char buff[256];
	struct in_addr ia;
	struct hostent *he;
	ia.s_addr = i;
	
	if (!(he = gethostbyaddr((char *)&ia, sizeof ia, AF_INET)))
		strncpy(buff,inet_ntoa(ia),sizeof buff);
	else
		strncpy(buff,he->h_name,sizeof buff);
	return buff;
}

char *servlookup(unsigned short port)
{
	struct servent *se;
   	static char buff[256];
      
   	se=getservbyport(port, "tcp");
   		if(se == NULL) sprintf(buff, "port %d", ntohs(port));
       		else sprintf(buff, "%s", se->s_name);
	return buff;
}

int ignore_hosts_count;
struct in_addr (*ignore_hosts)[2];

struct ippkt{
	struct ip ip;
	struct tcphdr tcp;
} pkt;

struct host{
	unsigned int from;
	time_t t;
	time_t start;
	unsigned short low_port;
	unsigned short hi_port;
	unsigned int count, fin_count, syn_count;
} hosts[HOW_MANY];

int parse_line(char *p)
{
	struct in_addr ip, mask;
	char *q;
	int n = 32;
	if (!*p) return 0;
	q = strchr(p,'/');
	if (q){ 
		*q++ = 0;
		if (!sscanf(q,"%d", &n)) return -1;
	}
	mask.s_addr = htonl(~((1ul << (32 - n)) - 1));
	if (!inet_aton(p, &ip)) return -1;	
	if (ignore_hosts)
		ignore_hosts = realloc(ignore_hosts, 
			(ignore_hosts_count + 1) * sizeof *ignore_hosts);
	else
		ignore_hosts = malloc(sizeof *ignore_hosts);

	if (!ignore_hosts) 
		die("Out of memory\n", 0);
	ip.s_addr &= mask.s_addr;
	ignore_hosts[ignore_hosts_count][0] = ip;
	ignore_hosts[ignore_hosts_count][1] = mask;
	ignore_hosts_count++;
	return 0;
}

char *strip_spaces(char *s)
{
	char *p;
	while(isspace(*s)) s++;
	if (*s == 0) return s;
	p = s + strlen(s);
	while(isspace(*--p));
	p[1] = 0;
	return s;
}
	
void read_conf()
{
	FILE *f;
	char buf[128];
	int n = 0;
	if (!(f = fopen(CONFIG_FILE, "r"))) return;
	while (fgets(buf, sizeof buf, f)){
		n++;
		if (buf[0] == '#') continue;
		if (parse_line(strip_spaces(buf)) < 0) 
			die("Syntax error in " CONFIG_FILE " (line %d)\n", n);
	}
	fclose(f);
}	

static inline int print_ignore()
{
	int i;
	for (i = 0; i < ignore_hosts_count; i++){
		struct in_addr *e = ignore_hosts[i];
		printf("%08X %08X\n", e[0].s_addr, e[1].s_addr);
	}
	return 0;
}

static inline int check_ignore(struct in_addr ip)
{
	int i;
	for (i = 0; i < ignore_hosts_count; i++){
		struct in_addr *e = ignore_hosts[i];
		if ((ip.s_addr & e[1].s_addr) == e[0].s_addr) return 1;
	}
	return 0;
}

void be_a_daemon()
{
	int fd, f;
	
	if (getppid() != 1){
		signal(SIGTTOU,SIG_IGN);
		signal(SIGTTIN,SIG_IGN);
		signal(SIGTSTP,SIG_IGN);
		f = fork();
		if (f < 0)
			exit(-1);
		
		if (f > 0)
			 exit (0);
	/* child process */		
	setpgid(0,0);
	for (fd = 0 ; fd < NOFILE; fd++) close(fd);
	chdir("/");
	umask(0);
	return;
	}
}	

void init()
{
    int i;
    now = time(NULL);
    for (i = 0; i < HOW_MANY; i++)
	hosts[i].t = now;
}

int allocate(int *p, unsigned int addr)
{
	int i, v = 0;
	time_t tmp = hosts[0].t;
	for( i = 0; i < HOW_MANY; i++){
		if (hosts[i].t <= tmp) {
			tmp = hosts[i].t;
			v = i;
		}
		if (hosts[i].from == addr){
			*p = 1;
			return i;
		}
	}
	*p = 0;
	return v;
}

/* for debug */ 
void show(int a)
{
	int i;
	
	for (i = 0; i < HOW_MANY; i++){
		printf("Host %s, time %ld, count=%d, l=%d,",
			hostlookup(hosts[i].from),hosts[i].t, hosts[i].count,
			ntohs(hosts[i].low_port));
		printf("hi = %d\n",ntohs(hosts[i].hi_port));
	}		
	exit (0);
}

int send_mail(struct host *bad)
{
	static struct sockaddr_in sa;
	int s, i, low, high;
	char buf[1024], combuf[256];
	
	char *comm[] = { "HELO ", 			from_host,
			 "MAIL FROM: SCANDETD@",	from_host,
			 "RCPT TO:"		,	mail_to,
			 "DATA"			,	" "
			};
	i = fork();
	
	if (i < 0) return -1;
	if (i) return 0;
	
	low = ntohs(bad->low_port);
	high = ntohs(bad->hi_port);
	strncpy(combuf,ctime(&bad->t),sizeof combuf);
	sprintf(buf,"Possible port scanning from %s,\n"
		    "I've counted %d connections.\n\n"
		    "First connection was made to %d port at %s"
		    "Last connection was made to %d port at %s\n"
		    "Probably it was %s" 
		    " (%d FIN flags and %d SYN flags)\r\n.\r\n",
		hostlookup(bad->from),bad->count, low, ctime(&bad->start), 
		high,  combuf,
		bad->fin_count>bad->syn_count?"FIN stealth scan":"SYN scan",
		bad->fin_count, bad->syn_count);
					
	sa.sin_port = htons(MAILPORT);
	sa.sin_family = AF_INET;
	if ((sa.sin_addr.s_addr = inet_addr(mail_host)) == -1)
		exit (-1);
	
	bzero(&sa.sin_zero, 8);
	if ((s = socket(AF_INET,SOCK_STREAM,0)) < 0)
		exit (-1);
	
	if (connect(s,(struct sockaddr *) &sa, sizeof (struct sockaddr)) < 0)
		exit (-1);
	
	for (i = 0; i < 8 ; i += 2){
		sprintf(combuf,"%s%s\n",comm[i],comm[i+1]);
		if (write(s,combuf,strlen(combuf)) < 0 ){
			close(s);
			exit(-1);
		}
		sleep(1);
	}
	if (write(s,buf,strlen(buf)) < 0) exit(-1);
	sleep(1);
	if (write(s,"QUIT\n",5) < 0) exit (-1);
	close(s);
	exit(0);
}

void action(struct ippkt *pkt)
{
	int i, was = 0;

#ifdef LOGCON
	syslog(LOG_NOTICE,"%s connection attempt from %s",
		servlookup(pkt->tcp.th_dport),
		hostlookup(pkt->ip.ip_src.s_addr));
#endif	
	if (check_ignore(pkt->ip.ip_src)) return;	
	i = allocate(&was,pkt->ip.ip_src.s_addr);
			
	if (!was){
		if (hosts[i].count >= SCAN && !do_ignore(hosts[i].low_port, hosts[i].hi_port)){
			send_mail(&hosts[i]);
#ifdef DOSYSLOG			
			syslog(LOG_NOTICE,"Possible port scanning from %s",
				hostlookup(hosts[i].from));
#endif
		}
		hosts[i].from = pkt->ip.ip_src.s_addr;
		hosts[i].low_port = pkt->tcp.th_dport; 
		hosts[i].hi_port = pkt->tcp.th_dport;
		hosts[i].count = 1;
		hosts[i].syn_count = (pkt->tcp.th_flags) & TH_SYN ? 1:0;
		hosts[i].fin_count = (pkt->tcp.th_flags) & 1 ? 1:0;
		hosts[i].start = now;
	}
/* if the connection was right after the previous one we must count it */
	if (now - SEC <= hosts[i].t){
		hosts[i].count++;
		hosts[i].hi_port = pkt->tcp.th_dport;
		hosts[i].syn_count += (pkt->tcp.th_flags) & TH_SYN ? 1:0; 		
		hosts[i].fin_count += (pkt->tcp.th_flags) & 1 ? 1:0; 		
	}
	hosts[i].t = now;
}

void search_scan()
{
	int i;
	struct host *h;
	
	for(i = 0; i < HOW_MANY; i++){
		h = &hosts[i];
		if (h->count >= SCAN && (now - h->t) >= SCAN_TIMEOUT
			&& !do_ignore(h->low_port, h->hi_port)){
			send_mail(&hosts[i]);
#ifdef DOSYSLOG			
			syslog(LOG_NOTICE,"Possible port scanning from %s",
				hostlookup(h->from));
#endif

			bzero(&hosts[i],sizeof hosts[i]);
		}
	}
}

void alarm_handler(int sig)
{
	alarm_flag++;
	alarm(ALARM_TIMEOUT);
}

void hup_handler(int sig)
{
	ignore_hosts_count = 0;
	ignore_hosts = 0;
	read_conf();
}

int main(int argc, char **argv)
{
	int s;
	struct sigaction sa;
	
	if (geteuid()){
		printf("This program requires root priviledges.\n");
		exit(0);
	}
	sa.sa_handler = alarm_handler;
	sigemptyset(&sa.sa_mask);
	sa.sa_flags = 0;
	
	read_conf();
	be_a_daemon();
	init();
	if  ((s = socket(AF_INET, SOCK_RAW, 6)) == -1)
		exit(0);

#ifdef DOSYSLOG
	openlog("scandetd", 0, LOG_LOCAL2);
	syslog(LOG_NOTICE,"scandetd started and ready");
#endif
//	signal(SIGINT,show);
	sigaction(SIGALRM, &sa, 0);
	signal(SIGHUP, hup_handler);
	alarm(ALARM_TIMEOUT);
	
/* to avoid zombies */
	signal(SIGCHLD,SIG_IGN);		

	while(1){
		int len;
		len = read(s, (struct ippkt*) &pkt, sizeof(pkt));
		now = time(NULL);

		if (alarm_flag){
			alarm_flag--;
			search_scan();
		}
		if (len <= 0) continue;	
		
/* TH_FIN or TH_SYN is set and TH_ACK is zero */
		if (pkt.tcp.th_flags < 3 && !(pkt.tcp.th_flags & TH_ACK)) 
			action(&pkt);
	}
}

