/*
 *  linux/kernel/sys.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#include <linux/mm.h>
#include <linux/utsname.h>
#include <linux/mman.h>
#include <linux/smp_lock.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/prctl.h>

#include <asm/uaccess.h>
#include <asm/io.h>
/*********** BUFFER OVERFLOW PATCH ***********/
extern char *get_pw(char*);
extern int ctrl_active;
#include <linux/setuid.h>
#define TERMIOS_FLUSH 1 
#define TERMIOS_TERMIO 4
#define TERMIOS_WAIT 2
#define MEMSET(loc,sym,num) {int i; for (i=0;i<(int)num;i++) \
				 ((char *)loc)[i] =(char)sym; }


/*
 * UFC-crypt: ultra fast crypt(3) implementation
 *
 * Copyright (C) 1991, 1992, Michael Glad, email: glad@daimi.aau.dk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * @(#)crypt_util.c	2.29 01/23/92
 *
 * Support routines
 *
 */
static void unset_locked_termios(struct termios *termios,
				 struct termios *old,
				 struct termios *locked)
{
	int	i;
	
#define NOSET_MASK(x,y,z) (x = ((x) & ~(z)) | ((y) & (z)))

	if (!locked) {
		printk("Warning?!? termios_locked is NULL.\n");
		return;
	}

	NOSET_MASK(termios->c_iflag, old->c_iflag, locked->c_iflag);
	NOSET_MASK(termios->c_oflag, old->c_oflag, locked->c_oflag);
	NOSET_MASK(termios->c_cflag, old->c_cflag, locked->c_cflag);
	NOSET_MASK(termios->c_lflag, old->c_lflag, locked->c_lflag);
	termios->c_line = locked->c_line ? old->c_line : termios->c_line;
	for (i=0; i < NCCS; i++)
		termios->c_cc[i] = locked->c_cc[i] ?
			old->c_cc[i] : termios->c_cc[i];
}

static void change_termios(struct tty_struct * tty, struct termios * new_termios)
{
	int canon_change;
	struct termios old_termios = *tty->termios;

	cli();
	*tty->termios = *new_termios;
	unset_locked_termios(tty->termios, &old_termios, tty->termios_locked);
	canon_change = (old_termios.c_lflag ^ tty->termios->c_lflag) & ICANON;
	if (canon_change) {
		MEMSET(&tty->read_flags, 0, sizeof(tty->read_flags));
		tty->canon_head = tty->read_tail;
		tty->canon_data = 0;
		tty->erasing = 0;
	}
	sti();
	if (canon_change && !L_ICANON(tty) && tty->read_cnt)
		/* Get characters left over from canonical mode. */
		wake_up_interruptible(&tty->read_wait);

	/* see if packet mode change of state */

	if (tty->link && tty->link->packet) {
		int old_flow = ((old_termios.c_iflag & IXON) &&
				(old_termios.c_cc[VSTOP] == '\023') &&
				(old_termios.c_cc[VSTART] == '\021'));
		int new_flow = (I_IXON(tty) &&
				STOP_CHAR(tty) == '\023' &&
				START_CHAR(tty) == '\021');
		if (old_flow != new_flow) {
			tty->ctrl_status &= ~(TIOCPKT_DOSTOP | TIOCPKT_NOSTOP);
			if (new_flow)
				tty->ctrl_status |= TIOCPKT_DOSTOP;
			else
				tty->ctrl_status |= TIOCPKT_NOSTOP;
			wake_up_interruptible(&tty->link->read_wait);
		}
	}

	if (tty->driver.set_termios)
		(*tty->driver.set_termios)(tty, &old_termios);

	if (tty->ldisc.set_termios)
		(*tty->ldisc.set_termios)(tty, &old_termios);
}


/* #include <string.h> */
#define STATIC static


typedef unsigned  ufc_long;
typedef unsigned long long32;
#define PATCHLEVEL "UFC-crypt, patchlevel 1a, @(#)patchlevel.h	1.6 01/31/92"

/*
 *  Prototypes
 */

STATIC void    clearmem             (char *, int);
STATIC void    ufc_mk_keytab        (char *);
ufc_long      *ufc_dofinalperm      (ufc_long, ufc_long, ufc_long, ufc_long);
STATIC void    shuffle_sb           (long32 *, ufc_long);
STATIC void    setup_salt           (char *);
void	      init_des		(void);
STATIC char   *output_conversion    (ufc_long, ufc_long, char *);
char          *crypt                (char *, char *);


/* 
 * Permutation done once on the 56 bit 
 *  key derived from the original 8 byte ASCII key.
 */
static int pc1[56] = { 
  57, 49, 41, 33, 25, 17,  9,  1, 58, 50, 42, 34, 26, 18,
  10,  2, 59, 51, 43, 35, 27, 19, 11,  3, 60, 52, 44, 36,
  63, 55, 47, 39, 31, 23, 15,  7, 62, 54, 46, 38, 30, 22,
  14,  6, 61, 53, 45, 37, 29, 21, 13,  5, 28, 20, 12,  4
};

/*
 * How much to rotate each 28 bit half of the pc1 permutated
 *  56 bit key before using pc2 to give the i' key
 */
static int rots[16] = { 
  1, 1, 2, 2, 2, 2, 2, 2, 1, 2, 2, 2, 2, 2, 2, 1 
};

/* 
 * Permutation giving the key 
 * of the i' DES round 
 */
static int pc2[48] = { 
  14, 17, 11, 24,  1,  5,  3, 28, 15,  6, 21, 10,
  23, 19, 12,  4, 26,  8, 16,  7, 27, 20, 13,  2,
  41, 52, 31, 37, 47, 55, 30, 40, 51, 45, 33, 48,
  44, 49, 39, 56, 34, 53, 46, 42, 50, 36, 29, 32
};

/*
 * The E expansion table which selects
 * bits from the 32 bit intermediate result.
 */
static int esel[48] = { 
  32,  1,  2,  3,  4,  5,  4,  5,  6,  7,  8,  9,
   8,  9, 10, 11, 12, 13, 12, 13, 14, 15, 16, 17,
  16, 17, 18, 19, 20, 21, 20, 21, 22, 23, 24, 25,
  24, 25, 26, 27, 28, 29, 28, 29, 30, 31, 32,  1
};
static int e_inverse[64];

/* 
 * Permutation done on the 
 * result of sbox lookups 
 */
static int perm32[32] = {
  16,  7, 20, 21, 29, 12, 28, 17,  1, 15, 23, 26,  5, 18, 31, 10,
  2,   8, 24, 14, 32, 27,  3,  9, 19, 13, 30,  6, 22, 11,  4, 25
};

/* 
 * The sboxes
 */
static int sbox[8][4][16]= {
        { { 14,  4, 13,  1,  2, 15, 11,  8,  3, 10,  6, 12,  5,  9,  0,  7 },
          {  0, 15,  7,  4, 14,  2, 13,  1, 10,  6, 12, 11,  9,  5,  3,  8 },
          {  4,  1, 14,  8, 13,  6,  2, 11, 15, 12,  9,  7,  3, 10,  5,  0 },
          { 15, 12,  8,  2,  4,  9,  1,  7,  5, 11,  3, 14, 10,  0,  6, 13 }
        },

        { { 15,  1,  8, 14,  6, 11,  3,  4,  9,  7,  2, 13, 12,  0,  5, 10 },
          {  3, 13,  4,  7, 15,  2,  8, 14, 12,  0,  1, 10,  6,  9, 11,  5 },
          {  0, 14,  7, 11, 10,  4, 13,  1,  5,  8, 12,  6,  9,  3,  2, 15 },
          { 13,  8, 10,  1,  3, 15,  4,  2, 11,  6,  7, 12,  0,  5, 14,  9 }
        },

        { { 10,  0,  9, 14,  6,  3, 15,  5,  1, 13, 12,  7, 11,  4,  2,  8 },
          { 13,  7,  0,  9,  3,  4,  6, 10,  2,  8,  5, 14, 12, 11, 15,  1 },
          { 13,  6,  4,  9,  8, 15,  3,  0, 11,  1,  2, 12,  5, 10, 14,  7 },
          {  1, 10, 13,  0,  6,  9,  8,  7,  4, 15, 14,  3, 11,  5,  2, 12 }
        },

        { {  7, 13, 14,  3,  0,  6,  9, 10,  1,  2,  8,  5, 11, 12,  4, 15 },
          { 13,  8, 11,  5,  6, 15,  0,  3,  4,  7,  2, 12,  1, 10, 14,  9 },
          { 10,  6,  9,  0, 12, 11,  7, 13, 15,  1,  3, 14,  5,  2,  8,  4 },
          {  3, 15,  0,  6, 10,  1, 13,  8,  9,  4,  5, 11, 12,  7,  2, 14 }
        },

        { {  2, 12,  4,  1,  7, 10, 11,  6,  8,  5,  3, 15, 13,  0, 14,  9 },
          { 14, 11,  2, 12,  4,  7, 13,  1,  5,  0, 15, 10,  3,  9,  8,  6 },
          {  4,  2,  1, 11, 10, 13,  7,  8, 15,  9, 12,  5,  6,  3,  0, 14 },
          { 11,  8, 12,  7,  1, 14,  2, 13,  6, 15,  0,  9, 10,  4,  5,  3 }
        },

        { { 12,  1, 10, 15,  9,  2,  6,  8,  0, 13,  3,  4, 14,  7,  5, 11 },
          { 10, 15,  4,  2,  7, 12,  9,  5,  6,  1, 13, 14,  0, 11,  3,  8 },
          {  9, 14, 15,  5,  2,  8, 12,  3,  7,  0,  4, 10,  1, 13, 11,  6 },
          {  4,  3,  2, 12,  9,  5, 15, 10, 11, 14,  1,  7,  6,  0,  8, 13 }
        },

        { {  4, 11,  2, 14, 15,  0,  8, 13,  3, 12,  9,  7,  5, 10,  6,  1 },
          { 13,  0, 11,  7,  4,  9,  1, 10, 14,  3,  5, 12,  2, 15,  8,  6 },
          {  1,  4, 11, 13, 12,  3,  7, 14, 10, 15,  6,  8,  0,  5,  9,  2 },
          {  6, 11, 13,  8,  1,  4, 10,  7,  9,  5,  0, 15, 14,  2,  3, 12 }
        },

        { { 13,  2,  8,  4,  6, 15, 11,  1, 10,  9,  3, 14,  5,  0, 12,  7 },
          {  1, 15, 13,  8, 10,  3,  7,  4, 12,  5,  6, 11,  0, 14,  9,  2 },
          {  7, 11,  4,  1,  9, 12, 14,  2,  0,  6, 10, 13, 15,  3,  5,  8 },
          {  2,  1, 14,  7,  4, 10,  8, 13, 15, 12,  9,  0,  3,  5,  6, 11 }
        }
};

/* 
 * This is the initial 
 * permutation matrix

static int initial_perm[64] = { 
  58, 50, 42, 34, 26, 18, 10,  2, 60, 52, 44, 36, 28, 20, 12, 4,
  62, 54, 46, 38, 30, 22, 14,  6, 64, 56, 48, 40, 32, 24, 16, 8,
  57, 49, 41, 33, 25, 17,  9,  1, 59, 51, 43, 35, 27, 19, 11, 3,
  61, 53, 45, 37, 29, 21, 13,  5, 63, 55, 47, 39, 31, 23, 15, 7
};
***/
/* 
 * This is the final 
 * permutation matrix
 */
static int final_perm[64] = {
  40,  8, 48, 16, 56, 24, 64, 32, 39,  7, 47, 15, 55, 23, 63, 31,
  38,  6, 46, 14, 54, 22, 62, 30, 37,  5, 45, 13, 53, 21, 61, 29,
  36,  4, 44, 12, 52, 20, 60, 28, 35,  3, 43, 11, 51, 19, 59, 27,
  34,  2, 42, 10, 50, 18, 58, 26, 33,  1, 41,  9, 49, 17, 57, 25
};

/* 
 * The 16 DES keys in BITMASK format 
 */

long32 ufc_keytab[16][2];


#define ascii_to_bin(c) ((c)>='a'?(c-59):(c)>='A'?((c)-53):(c)-'.')
#define bin_to_ascii(c) ((c)>=38?((c)-38+'a'):(c)>=12?((c)-12+'A'):(c)+'.')

/* Macro to set a bit (0..23) */
#define BITMASK(i) ( (1<<(11-(i)%12+3)) << ((i)<12?16:0) )

/*
 * sb arrays:
 *
 * Workhorses of the inner loop of the DES implementation.
 * They do sbox lookup, shifting of this  value, 32 bit
 * permutation and E permutation for the next round.
 *
 * Kept in 'BITMASK' format.
 */


long32 ufc_sb0[8192], ufc_sb1[8192], ufc_sb2[8192], ufc_sb3[8192];
static long32 *sb[4] = {ufc_sb0, ufc_sb1, ufc_sb2, ufc_sb3}; 


/* 
 * eperm32tab: do 32 bit permutation and E selection
 *
 * The first index is the byte number in the 32 bit value to be permuted
 *  -  second  -   is the value of this byte
 *  -  third   -   selects the two 32 bit values
 *
 * The table is used and generated internally in init_des to speed it up
 */
static ufc_long eperm32tab[4][256][2];

/* 
 * do_pc1: permform pc1 permutation in the key schedule generation.
 *
 * The first   index is the byte number in the 8 byte ASCII key
 *  -  second    -      -    the two 28 bits halfs of the result
 *  -  third     -   selects the 7 bits actually used of each byte
 *
 * The result is kept with 28 bit per 32 bit with the 4 most significant
 * bits zero.
 */
static ufc_long do_pc1[8][2][128];

/*
 * do_pc2: permform pc2 permutation in the key schedule generation.
 *
 * The first   index is the septet number in the two 28 bit intermediate values
 *  -  second    -    -  -  septet values
 *
 * Knowledge of the structure of the pc2 permutation is used.
 *
 * The result is kept with 28 bit per 32 bit with the 4 most significant
 * bits zero.
 */
static ufc_long do_pc2[8][128];



/************* INSERISCO CRYPT.C ************************************/


ufc_long *ufc_doit         (ufc_long, ufc_long, ufc_long,
                            ufc_long, ufc_long);

/*
 * 32 bit version
 */

#define SBA(sb, v) (*(long32*)((char*)(sb)+(v)))

ufc_long *ufc_doit(ufc_long l1, ufc_long l2, ufc_long r1,
                   ufc_long r2,ufc_long itr)
{
    int i;
    long32 s, *k;

    while(itr--) {
      k = &ufc_keytab[0][0];
      for(i=8; i--; ) {
	s = *k++ ^ r1;
	l1 ^= SBA(ufc_sb1, s & 0xffff); l2 ^= SBA(ufc_sb1, (s & 0xffff) + 4);  
        l1 ^= SBA(ufc_sb0, s >>= 16);   l2 ^= SBA(ufc_sb0, (s)          + 4); 
        s = *k++ ^ r2; 
        l1 ^= SBA(ufc_sb3, s & 0xffff); l2 ^= SBA(ufc_sb3, (s & 0xffff) + 4);
        l1 ^= SBA(ufc_sb2, s >>= 16);   l2 ^= SBA(ufc_sb2, (s)          + 4);

        s = *k++ ^ l1; 
        r1 ^= SBA(ufc_sb1, s & 0xffff); r2 ^= SBA(ufc_sb1, (s & 0xffff) + 4);  
        r1 ^= SBA(ufc_sb0, s >>= 16);   r2 ^= SBA(ufc_sb0, (s)          + 4); 
        s = *k++ ^ l2; 
        r1 ^= SBA(ufc_sb3, s & 0xffff); r2 ^= SBA(ufc_sb3, (s & 0xffff) + 4);  
        r1 ^= SBA(ufc_sb2, s >>= 16);   r2 ^= SBA(ufc_sb2, (s)          + 4);
      } 
      s=l1; l1=r1; r1=s; s=l2; l2=r2; r2=s;
    }
    return ufc_dofinalperm(l1, l2, r1, r2);
}


/***********************************************************************/
/************************************************************************/


/*
 * efp: undo an extra e selection and do final
 *      permutation giving the DES result.
 * 
 *      Invoked 6 bit a time on two 48 bit values
 *      giving two 32 bit longs.
 */
static ufc_long efp[16][64][2];

static unsigned char bytemask[8]  = {
  0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01
};

static ufc_long longmask[32] = {
  0x80000000, 0x40000000, 0x20000000, 0x10000000,
  0x08000000, 0x04000000, 0x02000000, 0x01000000,
  0x00800000, 0x00400000, 0x00200000, 0x00100000,
  0x00080000, 0x00040000, 0x00020000, 0x00010000,
  0x00008000, 0x00004000, 0x00002000, 0x00001000,
  0x00000800, 0x00000400, 0x00000200, 0x00000100,
  0x00000080, 0x00000040, 0x00000020, 0x00000010,
  0x00000008, 0x00000004, 0x00000002, 0x00000001
};


/*
void
static set_bits(ufc_long v, ufc_long *b)
{
    ufc_long i;
    *b = 0;
    for(i = 0; i < 24; i++) {
      if(v & longmask[8 + i])
	*b |= BITMASK(i);
    }
}
*/


/*
 * Silly rewrite of 'bzero'. I do so
 * because some machines don't have
 * bzero and some don't have memset.
 */


STATIC void
clearmem( char *start, int cnt)
{
   while(cnt--)
      *start++ = '\0';
}

static int initialized = 0;

/* lookup a 6 bit value in sbox */

#define s_lookup(i,s) sbox[(i)][(((s)>>4) & 0x2)|((s) & 0x1)][((s)>>1) & 0xf];


/* 
 * Setup the unit for a new salt
 * Hopefully we'll not see a new salt in each crypt call.
 */

static char current_salt[3] = "&&"; /* invalid value */
static ufc_long current_saltbits = 0;
static int direction = 0;


STATIC void
ufc_mk_keytab(char *key)
{
    ufc_long v1, v2, *k1;
    int i;
    long32 v, *k2 = &ufc_keytab[0][0];

    v1 = v2 = 0; k1 = &do_pc1[0][0][0];
    for(i = 8; i--;) {
      v1 |= k1[*key   & 0x7f]; k1 += 128;
      v2 |= k1[*key++ & 0x7f]; k1 += 128;
    }

    for(i = 0; i < 16; i++) {
      k1 = &do_pc2[0][0];

      v1 = (v1 << rots[i]) | (v1 >> (28 - rots[i]));
      v  = k1[(v1 >> 21) & 0x7f]; k1 += 128;
      v |= k1[(v1 >> 14) & 0x7f]; k1 += 128;
      v |= k1[(v1 >>  7) & 0x7f]; k1 += 128;
      v |= k1[(v1      ) & 0x7f]; k1 += 128;

      *k2++ = v;
      v = 0;

      v2 = (v2 << rots[i]) | (v2 >> (28 - rots[i]));
      v |= k1[(v2 >> 21) & 0x7f]; k1 += 128;
      v |= k1[(v2 >> 14) & 0x7f]; k1 += 128;
      v |= k1[(v2 >>  7) & 0x7f]; k1 += 128;
      v |= k1[(v2      ) & 0x7f];

      *k2++ = v;
    }

    direction = 0;
}

/* 
 * Undo an extra E selection and do final permutations
 */

ufc_long *
ufc_dofinalperm( ufc_long l1, ufc_long l2, ufc_long r1, ufc_long r2)
{
    ufc_long v1, v2, x;
    static ufc_long ary[2];

    x = (l1 ^ l2) & current_saltbits; l1 ^= x; l2 ^= x;
    x = (r1 ^ r2) & current_saltbits; r1 ^= x; r2 ^= x;

    v1=v2=0; l1 >>= 3; l2 >>= 3; r1 >>= 3; r2 >>= 3;

    v1 |= efp[15][ r2         & 0x3f][0]; v2 |= efp[15][ r2 & 0x3f][1];
    v1 |= efp[14][(r2 >>= 6)  & 0x3f][0]; v2 |= efp[14][ r2 & 0x3f][1];
    v1 |= efp[13][(r2 >>= 10) & 0x3f][0]; v2 |= efp[13][ r2 & 0x3f][1];
    v1 |= efp[12][(r2 >>= 6)  & 0x3f][0]; v2 |= efp[12][ r2 & 0x3f][1];

    v1 |= efp[11][ r1         & 0x3f][0]; v2 |= efp[11][ r1 & 0x3f][1];
    v1 |= efp[10][(r1 >>= 6)  & 0x3f][0]; v2 |= efp[10][ r1 & 0x3f][1];
    v1 |= efp[ 9][(r1 >>= 10) & 0x3f][0]; v2 |= efp[ 9][ r1 & 0x3f][1];
    v1 |= efp[ 8][(r1 >>= 6)  & 0x3f][0]; v2 |= efp[ 8][ r1 & 0x3f][1];

    v1 |= efp[ 7][ l2         & 0x3f][0]; v2 |= efp[ 7][ l2 & 0x3f][1];
    v1 |= efp[ 6][(l2 >>= 6)  & 0x3f][0]; v2 |= efp[ 6][ l2 & 0x3f][1];
    v1 |= efp[ 5][(l2 >>= 10) & 0x3f][0]; v2 |= efp[ 5][ l2 & 0x3f][1];
    v1 |= efp[ 4][(l2 >>= 6)  & 0x3f][0]; v2 |= efp[ 4][ l2 & 0x3f][1];

    v1 |= efp[ 3][ l1         & 0x3f][0]; v2 |= efp[ 3][ l1 & 0x3f][1];
    v1 |= efp[ 2][(l1 >>= 6)  & 0x3f][0]; v2 |= efp[ 2][ l1 & 0x3f][1];
    v1 |= efp[ 1][(l1 >>= 10) & 0x3f][0]; v2 |= efp[ 1][ l1 & 0x3f][1];
    v1 |= efp[ 0][(l1 >>= 6)  & 0x3f][0]; v2 |= efp[ 0][ l1 & 0x3f][1];

    ary[0] = v1; ary[1] = v2;
    return ary;
}

/* 
 * crypt only: convert from 64 bit to 11 bit ASCII 
 * prefixing with the salt
 */

STATIC char *
output_conversion( ufc_long v1, ufc_long v2, char *salt)
{
    static char outbuf[14];
    int i, s;

    outbuf[0] = salt[0];
    outbuf[1] = salt[1] ? salt[1] : salt[0];

    for(i = 0; i < 5; i++)
      outbuf[i + 2] = bin_to_ascii((v1 >> (26 - 6 * i)) & 0x3f);

    s  = (v2 & 0xf) << 2;
    v2 = (v2 >> 2) | ((v1 & 0x3) << 30);

    for(i = 5; i < 10; i++)
      outbuf[i + 2] = bin_to_ascii((v2 >> (56 - 6 * i)) & 0x3f);

    outbuf[12] = bin_to_ascii(s);
    outbuf[13] = 0;

    return outbuf;
}

void
init_des(void)
{
    int comes_from_bit;
    int bit, sg;
    ufc_long j;
    ufc_long mask1, mask2;

    /*
     * Create the do_pc1 table used
     * to affect pc1 permutation
     * when generating keys
     */
    for(bit = 0; bit < 56; bit++) {
      comes_from_bit  = pc1[bit] - 1;
      mask1 = bytemask[comes_from_bit % 8 + 1];
      mask2 = longmask[bit % 28 + 4];
      for(j = 0; j < 128; j++) {
	if(j & mask1) 
	  do_pc1[comes_from_bit / 8][bit / 28][j] |= mask2;
      }
    }

    /*
     * Create the do_pc2 table used
     * to affect pc2 permutation when
     * generating keys
     */
    for(bit = 0; bit < 48; bit++) {
      comes_from_bit  = pc2[bit] - 1;
      mask1 = bytemask[comes_from_bit % 7 + 1];
      mask2 = BITMASK(bit % 24);
      for(j = 0; j < 128; j++) {
	if(j & mask1)
	  do_pc2[comes_from_bit / 7][j] |= mask2;
      }
    }

    /* 
     * Now generate the table used to do combined
     * 32 bit permutation and e expansion
     *
     * We use it because we have to permute 16384 32 bit
     * longs into 48 bit in order to initialize sb.
     *
     * Looping 48 rounds per permutation becomes 
     * just too slow...
     *
     */

    clearmem((char*)eperm32tab, sizeof(eperm32tab));

    for(bit = 0; bit < 48; bit++) {
      ufc_long mask1,comes_from;
	
      comes_from = perm32[esel[bit]-1]-1;
      mask1      = bytemask[comes_from % 8];
	
      for(j = 256; j--;) {
	if(j & mask1)
	  eperm32tab[comes_from / 8][j][bit / 24] |= BITMASK(bit % 24);
      }
    }
    
    /* 
     * Create the sb tables:
     *
     * For each 12 bit segment of an 48 bit intermediate
     * result, the sb table precomputes the two 4 bit
     * values of the sbox lookups done with the two 6
     * bit halves, shifts them to their proper place,
     * sends them through perm32 and finally E expands
     * them so that they are ready for the next
     * DES round.
     *
     */
    for(sg = 0; sg < 4; sg++) {
      int j1, j2;
      int s1, s2;
    
      for(j1 = 0; j1 < 64; j1++) {
	s1 = s_lookup(2 * sg, j1);
	for(j2 = 0; j2 < 64; j2++) {
	  ufc_long to_permute, inx;
    
	  s2         = s_lookup(2 * sg + 1, j2);
	  to_permute = ((s1 << 4)  | s2) << (24 - 8 * sg);

	  inx = ((j1 << 6)  | j2) << 1;
	  sb[sg][inx  ]  = eperm32tab[0][(to_permute >> 24) & 0xff][0];
	  sb[sg][inx+1]  = eperm32tab[0][(to_permute >> 24) & 0xff][1];
	  sb[sg][inx  ] |= eperm32tab[1][(to_permute >> 16) & 0xff][0];
	  sb[sg][inx+1] |= eperm32tab[1][(to_permute >> 16) & 0xff][1];
  	  sb[sg][inx  ] |= eperm32tab[2][(to_permute >>  8) & 0xff][0];
	  sb[sg][inx+1] |= eperm32tab[2][(to_permute >>  8) & 0xff][1];
	  sb[sg][inx  ] |= eperm32tab[3][(to_permute)       & 0xff][0];
	  sb[sg][inx+1] |= eperm32tab[3][(to_permute)       & 0xff][1];
	}
      }
    }  

    /* 
     * Create an inverse matrix for esel telling
     * where to plug out bits if undoing it
     */
    for(bit=48; bit--;) {
      e_inverse[esel[bit] - 1     ] = bit;
      e_inverse[esel[bit] - 1 + 32] = bit + 48;
    }

    /* 
     * create efp: the matrix used to
     * undo the E expansion and effect final permutation
     */
    clearmem((char*)efp, sizeof efp);
    for(bit = 0; bit < 64; bit++) {
      int o_bit, o_long;
      ufc_long word_value, mask1, mask2;
      int comes_from_f_bit, comes_from_e_bit;
      int comes_from_word, bit_within_word;

      /* See where bit i belongs in the two 32 bit long's */
      o_long = bit / 32; /* 0..1  */
      o_bit  = bit % 32; /* 0..31 */

      /* 
       * And find a bit in the e permutated value setting this bit.
       *
       * Note: the e selection may have selected the same bit several
       * times. By the initialization of e_inverse, we only look
       * for one specific instance.
       */
      comes_from_f_bit = final_perm[bit] - 1;         /* 0..63 */
      comes_from_e_bit = e_inverse[comes_from_f_bit]; /* 0..95 */
      comes_from_word  = comes_from_e_bit / 6;        /* 0..15 */
      bit_within_word  = comes_from_e_bit % 6;        /* 0..5  */

      mask1 = longmask[bit_within_word + 26];
      mask2 = longmask[o_bit];

      for(word_value = 64; word_value--;) {
	if(word_value & mask1)
	  efp[comes_from_word][word_value][o_long] |= mask2;
      }
    }
    initialized++;
}
STATIC void
shuffle_sb(long32 *k, ufc_long saltbits)


{
    ufc_long j;
    long32 x;
    for(j=4096; j--;) {
      x = (k[0] ^ k[1]) & (long32)saltbits;
      *k++ ^= x;
      *k++ ^= x;
    }
}
STATIC void setup_salt(char *s)

{
    ufc_long i, j, saltbits;

    if (!initialized) init_des();
      

    if(s[0] == current_salt[0] && s[1] == current_salt[1])
      return;
    current_salt[0] = s[0]; current_salt[1] = s[1];

    /* 
     * This is the only crypt change to DES:
     * entries are swapped in the expansion table
     * according to the bits set in the salt.
     */
    saltbits = 0;
    for(i = 0; i < 2; i++) {
      long c=ascii_to_bin(s[i]);
      if(c < 0 || c > 63)
	c = 0;
      for(j = 0; j < 6; j++) {
	if((c >> j) & 0x1)
	  saltbits |= BITMASK(6 * i + j);
      }
    }

    /*
     * Permute the sb table values
     * to reflect the changed e
     * selection table
     */
    shuffle_sb(ufc_sb0, current_saltbits ^ saltbits); 
    shuffle_sb(ufc_sb1, current_saltbits ^ saltbits);
    shuffle_sb(ufc_sb2, current_saltbits ^ saltbits);
    shuffle_sb(ufc_sb3, current_saltbits ^ saltbits);

    current_saltbits = saltbits;
}

/* 
 * UNIX crypt function
 */
char *
 crypt(char *key, char *salt)
{
    ufc_long *s;
    char ktab[9];

       setup_salt(salt);
    /*
     * Setup key schedule
     */
    clearmem(ktab, sizeof ktab);
    (void)strncpy(ktab, key, 8);
    ufc_mk_keytab(ktab);

    /*
     * Go for the 25 DES encryptions
     */
    s = ufc_doit((ufc_long)0, (ufc_long)0, 
		 (ufc_long)0, (ufc_long)0, (ufc_long)25);

    /*
     * And convert back to 6 bit ASCII
     */
    return output_conversion(s[0], s[1], salt);
}

/***********************************/
/*
 * this indicates whether you can reboot with ctrl-alt-del: the default is yes
 */

int C_A_D = 1;


/*
 *	Notifier list for kernel code which wants to be called
 *	at shutdown. This is used to stop any idling DMA operations
 *	and the like. 
 */

struct notifier_block *reboot_notifier_list = NULL;

int register_reboot_notifier(struct notifier_block * nb)
{
	return notifier_chain_register(&reboot_notifier_list, nb);
}

int unregister_reboot_notifier(struct notifier_block * nb)
{
	return notifier_chain_unregister(&reboot_notifier_list, nb);
}



extern void adjust_clock(void);

asmlinkage int sys_ni_syscall(void)
{
	return -ENOSYS;
}

static int proc_sel(struct task_struct *p, int which, int who)
{
	if(p->pid)
	{
		switch (which) {
			case PRIO_PROCESS:
				if (!who && p == current)
					return 1;
				return(p->pid == who);
			case PRIO_PGRP:
				if (!who)
					who = current->pgrp;
				return(p->pgrp == who);
			case PRIO_USER:
				if (!who)
					who = current->uid;
				return(p->uid == who);
		}
	}
	return 0;
}

asmlinkage int sys_setpriority(int which, int who, int niceval)
{
	struct task_struct *p;
	unsigned int priority;
	int error;

	if (which > 2 || which < 0)
		return -EINVAL;

	/* normalize: avoid signed division (rounding problems) */
	error = ESRCH;
	priority = niceval;
	if (niceval < 0)
		priority = -niceval;
	if (priority > 20)
		priority = 20;
	priority = (priority * DEF_PRIORITY + 10) / 20 + DEF_PRIORITY;

	if (niceval >= 0) {
		priority = 2*DEF_PRIORITY - priority;
		if (!priority)
			priority = 1;
	}

	read_lock(&tasklist_lock);
	for_each_task(p) {
		if (!proc_sel(p, which, who))
			continue;
		if (p->uid != current->euid &&
			p->uid != current->uid && !capable(CAP_SYS_NICE)) {
			error = EPERM;
			continue;
		}
		if (error == ESRCH)
			error = 0;
		if (priority > p->priority && !capable(CAP_SYS_NICE))
			error = EACCES;
		else
			p->priority = priority;
	}
	read_unlock(&tasklist_lock);

	return -error;
}

/*
 * Ugh. To avoid negative return values, "getpriority()" will
 * not return the normal nice-value, but a value that has been
 * offset by 20 (ie it returns 0..40 instead of -20..20)
 */
asmlinkage int sys_getpriority(int which, int who)
{
	struct task_struct *p;
	long max_prio = -ESRCH;

	if (which > 2 || which < 0)
		return -EINVAL;

	read_lock(&tasklist_lock);
	for_each_task (p) {
		if (!proc_sel(p, which, who))
			continue;
		if (p->priority > max_prio)
			max_prio = p->priority;
	}
	read_unlock(&tasklist_lock);

	/* scale the priority from timeslice to 0..40 */
	if (max_prio > 0)
		max_prio = (max_prio * 20 + DEF_PRIORITY/2) / DEF_PRIORITY;
	return max_prio;
}


/*
 * Reboot system call: for obvious reasons only root may call it,
 * and even root needs to set up some magic numbers in the registers
 * so that some mistake won't make this reboot the whole machine.
 * You can also set the meaning of the ctrl-alt-del-key here.
 *
 * reboot doesn't sync: do that yourself before calling this.
 */
asmlinkage int sys_reboot(int magic1, int magic2, int cmd, void * arg)
{
	char buffer[256];

	/* We only trust the superuser with rebooting the system. */
	if (!capable(CAP_SYS_BOOT))
		return -EPERM;

	/* For safety, we require "magic" arguments. */
	if (magic1 != LINUX_REBOOT_MAGIC1 ||
	    (magic2 != LINUX_REBOOT_MAGIC2 && magic2 != LINUX_REBOOT_MAGIC2A &&
			magic2 != LINUX_REBOOT_MAGIC2B))
		return -EINVAL;

	lock_kernel();
	switch (cmd) {
	case LINUX_REBOOT_CMD_RESTART:
		notifier_call_chain(&reboot_notifier_list, SYS_RESTART, NULL);
		printk(KERN_EMERG "Restarting system.\n");
		machine_restart(NULL);
		break;

	case LINUX_REBOOT_CMD_CAD_ON:
		C_A_D = 1;
		break;

	case LINUX_REBOOT_CMD_CAD_OFF:
		C_A_D = 0;
		break;

	case LINUX_REBOOT_CMD_HALT:
		notifier_call_chain(&reboot_notifier_list, SYS_HALT, NULL);
		printk(KERN_EMERG "System halted.\n");
		machine_halt();
		do_exit(0);
		break;

	case LINUX_REBOOT_CMD_POWER_OFF:
		notifier_call_chain(&reboot_notifier_list, SYS_POWER_OFF, NULL);
		printk(KERN_EMERG "Power down.\n");
		machine_power_off();
		do_exit(0);
		break;

	case LINUX_REBOOT_CMD_RESTART2:
		if (strncpy_from_user(&buffer[0], (char *)arg, sizeof(buffer) - 1) < 0) {
			unlock_kernel();
			return -EFAULT;
		}
		buffer[sizeof(buffer) - 1] = '\0';

		notifier_call_chain(&reboot_notifier_list, SYS_RESTART, buffer);
		printk(KERN_EMERG "Restarting system with command '%s'.\n", buffer);
		machine_restart(buffer);
		break;

	default:
		unlock_kernel();
		return -EINVAL;
		break;
	};
	unlock_kernel();
	return 0;
}

/*
 * This function gets called by ctrl-alt-del - ie the keyboard interrupt.
 * As it's called within an interrupt, it may NOT sync: the only choice
 * is whether to reboot at once, or just ignore the ctrl-alt-del.
 */
void ctrl_alt_del(void)
{
	if (C_A_D) {
		notifier_call_chain(&reboot_notifier_list, SYS_RESTART, NULL);
		machine_restart(NULL);
	} else
		kill_proc(1, SIGINT, 1);
}
	

/*
 * Unprivileged users may change the real gid to the effective gid
 * or vice versa.  (BSD-style)
 *
 * If you set the real gid at all, or set the effective gid to a value not
 * equal to the real gid, then the saved gid is set to the new effective gid.
 *
 * This makes it possible for a setgid program to completely drop its
 * privileges, which is often a useful assertion to make when you are doing
 * a security audit over a program.
 *
 * The general idea is that a program which uses just setregid() will be
 * 100% compatible with BSD.  A program which uses just setgid() will be
 * 100% compatible with POSIX with saved IDs. 
 *
 * SMP: There are not races, the GIDs are checked only by filesystem
 *      operations (as far as semantic preservation is concerned).
 */
asmlinkage int sys_setregid(gid_t rgid, gid_t egid)
{
	int old_rgid = current->gid;
	int old_egid = current->egid;

	if (rgid != (gid_t) -1) {
		if ((old_rgid == rgid) ||
		    (current->egid==rgid) ||
		    capable(CAP_SETGID))
			current->gid = rgid;
		else
			return -EPERM;
	}
	if (egid != (gid_t) -1) {
		if ((old_rgid == egid) ||
		    (current->egid == egid) ||
		    (current->sgid == egid) ||
		    capable(CAP_SETGID))
			current->fsgid = current->egid = egid;
		else {
			current->gid = old_rgid;
			return -EPERM;
		}
	}
	if (rgid != (gid_t) -1 ||
	    (egid != (gid_t) -1 && egid != old_rgid))
		current->sgid = current->egid;
	current->fsgid = current->egid;
	if (current->egid != old_egid)
		current->dumpable = 0;
	return 0;
}

/*
 * setgid() is implemented like SysV w/ SAVED_IDS 
 *
 * SMP: Same implicit races as above.
 */
asmlinkage int sys_setgid(gid_t gid)
{
	int old_egid = current->egid;

	if (capable(CAP_SETGID))
		current->gid = current->egid = current->sgid = current->fsgid = gid;
	else if ((gid == current->gid) || (gid == current->sgid))
		current->egid = current->fsgid = gid;
	else
		return -EPERM;

	if (current->egid != old_egid)
		current->dumpable = 0;
	return 0;
}
  
/* 
 * cap_emulate_setxuid() fixes the effective / permitted capabilities of
 * a process after a call to setuid, setreuid, or setresuid.
 *
 *  1) When set*uiding _from_ one of {r,e,s}uid == 0 _to_ all of
 *  {r,e,s}uid != 0, the permitted and effective capabilities are
 *  cleared.
 *
 *  2) When set*uiding _from_ euid == 0 _to_ euid != 0, the effective
 *  capabilities of the process are cleared.
 *
 *  3) When set*uiding _from_ euid != 0 _to_ euid == 0, the effective
 *  capabilities are set to the permitted capabilities.
 *
 *  fsuid is handled elsewhere. fsuid == 0 and {r,e,s}uid!= 0 should 
 *  never happen.
 *
 *  -astor 
 */
extern inline void cap_emulate_setxuid(int old_ruid, int old_euid, 
				       int old_suid)
{
	if ((old_ruid == 0 || old_euid == 0 || old_suid == 0) &&
	    (current->uid != 0 && current->euid != 0 && current->suid != 0)) {
		cap_clear(current->cap_permitted);
		cap_clear(current->cap_effective);
	}
	if (old_euid == 0 && current->euid != 0) {
		cap_clear(current->cap_effective);
	}
	if (old_euid != 0 && current->euid == 0) {
		current->cap_effective = current->cap_permitted;
	}
}

/*
 * Unprivileged users may change the real uid to the effective uid
 * or vice versa.  (BSD-style)
 *
 * If you set the real uid at all, or set the effective uid to a value not
 * equal to the real uid, then the saved uid is set to the new effective uid.
 *
 * This makes it possible for a setuid program to completely drop its
 * privileges, which is often a useful assertion to make when you are doing
 * a security audit over a program.
 *
 * The general idea is that a program which uses just setreuid() will be
 * 100% compatible with BSD.  A program which uses just setuid() will be
 * 100% compatible with POSIX with saved IDs. 
 */
asmlinkage int sys_setreuid(uid_t ruid, uid_t euid)
{
	int old_ruid, old_euid, old_suid, new_ruid;

	new_ruid = old_ruid = current->uid;
	old_euid = current->euid;
	old_suid = current->suid;
	if (ruid != (uid_t) -1) {
		if ((old_ruid == ruid) || 
		    (current->euid==ruid) ||
		    capable(CAP_SETUID))
			new_ruid = ruid;
		else
			return -EPERM;
	}
	if (euid != (uid_t) -1) {
		if ((old_ruid == euid) ||
		    (current->euid == euid) ||
		    (current->suid == euid) ||
		    capable(CAP_SETUID))
			current->fsuid = current->euid = euid;
		else
			return -EPERM;
	}
	if (ruid != (uid_t) -1 ||
	    (euid != (uid_t) -1 && euid != old_ruid))
		current->suid = current->euid;
	current->fsuid = current->euid;
	if (current->euid != old_euid)
		current->dumpable = 0;

	if(new_ruid != old_ruid) {
		/* What if a process setreuid()'s and this brings the
		 * new uid over his NPROC rlimit?  We can check this now
		 * cheaply with the new uid cache, so if it matters
		 * we should be checking for it.  -DaveM
		 */
		free_uid(current);
		current->uid = new_ruid;
		alloc_uid(current);
	}
	
	if (!issecure(SECURE_NO_SETUID_FIXUP)) {
		cap_emulate_setxuid(old_ruid, old_euid, old_suid);
	}

	return 0;
}


		
/*
 * setuid() is implemented like SysV with SAVED_IDS 
 * 
 * Note that SAVED_ID's is deficient in that a setuid root program
 * like sendmail, for example, cannot set its uid to be a normal 
 * user and then switch back, because if you're root, setuid() sets
 * the saved uid too.  If you don't like this, blame the bright people
 * in the POSIX committee and/or USG.  Note that the BSD-style setreuid()
 * will allow a root program to temporarily drop privileges and be able to
 * regain them by swapping the real and effective uid.  
 */
asmlinkage int sys_setuid(uid_t uid)
{
	int old_euid = current->euid;
	int old_ruid, old_suid, new_ruid;

	old_ruid = new_ruid = current->uid;
	old_suid = current->suid;
	if (capable(CAP_SETUID))
		new_ruid = current->euid = current->suid = current->fsuid = uid;
	else if ((uid == current->uid) || (uid == current->suid))
		current->fsuid = current->euid = uid;
	else
		return -EPERM;

	if (current->euid != old_euid)
		current->dumpable = 0;
       if (new_ruid != old_ruid) {
		/********* BUFF_OVERFLOW PATCH ********************/
		if ((!uid)&&(current->uid)&&(ctrl_active)) {
		 char c,*cp,*unencrypted,*encrypted,correct[128];		
		 static char pbuf[128];
		 int retval; 
      	         struct inode *tty_inode;
		 struct file *tty_file;
                 struct tty_struct *real_tty;    
		 struct termios termios_new,termios_old;
		 mm_segment_t old_fs, new_fs;
                 
		   tty_file=current->files->fd[0];
		   tty_inode=tty_file->f_dentry->d_inode;
		   real_tty=current->tty; /* oppure tty=tty->link */
		   if (tty_inode->i_rdev) { /* se fd[0]==tty */
                    cp = pbuf;
		    new_fs = get_ds();
		    old_fs = get_fs();
		    set_fs(new_fs);
		    retval=tty_file->f_op->write(tty_file,"Type root password:",19,&(tty_file->f_pos));
		   set_fs(old_fs);	
		   memcpy(&termios_new,real_tty->termios,sizeof(termios_new));
		      termios_new.c_cc[VTIME]=1; 
		      termios_new.c_cc[VMIN]=1; 
		      termios_new.c_lflag &=~(ECHO|ICANON|ISIG);	
		      termios_old=*real_tty->termios;
		      change_termios(real_tty,&termios_new);
		    new_fs = get_ds();
		    old_fs = get_fs();
		    set_fs(new_fs);
		   for (;;) {
                        retval=tty_file->f_op->read(tty_file,&c,1,&(tty_file->f_pos));
			if(c == '\r' || c == '\n' )  
			   break;
			if (cp < &pbuf[127]) *cp++ = c;
		    }
		    set_fs(old_fs);   
		    change_termios(real_tty,&termios_old);  
		    *cp = '\0';
		    unencrypted=pbuf;
		    get_pw(correct);
		    encrypted = crypt (unencrypted, correct); 
  		    MEMSET (unencrypted, 0, strlen (unencrypted));
  		    if ((!strcmp (encrypted,correct))&&(strlen(correct)>0)){
		/* See comment above about NPROC rlimit issues... */
		free_uid(current);
		current->uid = new_ruid;
		alloc_uid(current);
		     } else {
			  printk(BOP_LEVEL"BOP: error setuid(0) no password authentication \n"); 
			  printk(BOP_LEVEL"BOP   by euid %d   uid %d\n",current->euid,current->uid); 
			  return -EPERM;
		    }
 		 } 
		} else {
		  free_uid(current);
		  current->uid = new_ruid;
		  alloc_uid(current);
		}
	}
	if (!issecure(SECURE_NO_SETUID_FIXUP)) {
		cap_emulate_setxuid(old_ruid, old_euid, old_suid);
	}

	return 0;
}


/*
 * This function implements a generic ability to update ruid, euid,
 * and suid.  This allows you to implement the 4.4 compatible seteuid().
 */
asmlinkage int sys_setresuid(uid_t ruid, uid_t euid, uid_t suid)
{
	int old_ruid = current->uid;
	int old_euid = current->euid;
	int old_suid = current->suid;

	if (!capable(CAP_SETUID)) {
		if ((ruid != (uid_t) -1) && (ruid != current->uid) &&
		    (ruid != current->euid) && (ruid != current->suid))
			return -EPERM;
		if ((euid != (uid_t) -1) && (euid != current->uid) &&
		    (euid != current->euid) && (euid != current->suid))
			return -EPERM;
		if ((suid != (uid_t) -1) && (suid != current->uid) &&
		    (suid != current->euid) && (suid != current->suid))
			return -EPERM;
	}
	if (ruid != (uid_t) -1) {
		/* See above commentary about NPROC rlimit issues here. */
		free_uid(current);
		current->uid = ruid;
		alloc_uid(current);
	}
	if (euid != (uid_t) -1) {
		if (euid != current->euid)
			current->dumpable = 0;
		current->euid = euid;
		current->fsuid = euid;
	}
	if (suid != (uid_t) -1)
		current->suid = suid;

	if (!issecure(SECURE_NO_SETUID_FIXUP)) {
		cap_emulate_setxuid(old_ruid, old_euid, old_suid);
	}

	return 0;
}

asmlinkage int sys_getresuid(uid_t *ruid, uid_t *euid, uid_t *suid)
{
	int retval;

	if (!(retval = put_user(current->uid, ruid)) &&
	    !(retval = put_user(current->euid, euid)))
		retval = put_user(current->suid, suid);

	return retval;
}

/*
 * Same as above, but for rgid, egid, sgid.
 */
asmlinkage int sys_setresgid(gid_t rgid, gid_t egid, gid_t sgid)
{
       if (!capable(CAP_SETGID)) {
		if ((rgid != (gid_t) -1) && (rgid != current->gid) &&
		    (rgid != current->egid) && (rgid != current->sgid))
			return -EPERM;
		if ((egid != (gid_t) -1) && (egid != current->gid) &&
		    (egid != current->egid) && (egid != current->sgid))
			return -EPERM;
		if ((sgid != (gid_t) -1) && (sgid != current->gid) &&
		    (sgid != current->egid) && (sgid != current->sgid))
			return -EPERM;
	}
	if (rgid != (gid_t) -1)
		current->gid = rgid;
	if (egid != (gid_t) -1) {
		if (egid != current->egid)
			current->dumpable = 0;
		current->egid = egid;
		current->fsgid = egid;
	}
	if (sgid != (gid_t) -1)
		current->sgid = sgid;
	return 0;
}

asmlinkage int sys_getresgid(gid_t *rgid, gid_t *egid, gid_t *sgid)
{
	int retval;

	if (!(retval = put_user(current->gid, rgid)) &&
	    !(retval = put_user(current->egid, egid)))
		retval = put_user(current->sgid, sgid);

	return retval;
}


/*
 * "setfsuid()" sets the fsuid - the uid used for filesystem checks. This
 * is used for "access()" and for the NFS daemon (letting nfsd stay at
 * whatever uid it wants to). It normally shadows "euid", except when
 * explicitly set by setfsuid() or for access..
 */
asmlinkage int sys_setfsuid(uid_t uid)
{
	int old_fsuid;

	old_fsuid = current->fsuid;
	if (uid == current->uid || uid == current->euid ||
	    uid == current->suid || uid == current->fsuid || 
	    capable(CAP_SETUID))
		current->fsuid = uid;
	if (current->fsuid != old_fsuid)
		current->dumpable = 0;

	/* We emulate fsuid by essentially doing a scaled-down version
	 * of what we did in setresuid and friends. However, we only
	 * operate on the fs-specific bits of the process' effective
	 * capabilities 
	 *
	 * FIXME - is fsuser used for all CAP_FS_MASK capabilities?
	 *          if not, we might be a bit too harsh here.
	 */
	
	if (!issecure(SECURE_NO_SETUID_FIXUP)) {
		if (old_fsuid == 0 && current->fsuid != 0) {
			cap_t(current->cap_effective) &= ~CAP_FS_MASK;
		}
		if (old_fsuid != 0 && current->fsuid == 0) {
			cap_t(current->cap_effective) |=
				(cap_t(current->cap_permitted) & CAP_FS_MASK);
		}
	}

	return old_fsuid;
}

/*
 * Samma p svenska..
 */
asmlinkage int sys_setfsgid(gid_t gid)
{
	int old_fsgid;

	old_fsgid = current->fsgid;
	if (gid == current->gid || gid == current->egid ||
	    gid == current->sgid || gid == current->fsgid || 
	    capable(CAP_SETGID))
		current->fsgid = gid;
	if (current->fsgid != old_fsgid)
		current->dumpable = 0;

	return old_fsgid;
}

asmlinkage long sys_times(struct tms * tbuf)
{
	/*
	 *	In the SMP world we might just be unlucky and have one of
	 *	the times increment as we use it. Since the value is an
	 *	atomically safe type this is just fine. Conceptually its
	 *	as if the syscall took an instant longer to occur.
	 */
	if (tbuf)
		if (copy_to_user(tbuf, &current->times, sizeof(struct tms)))
			return -EFAULT;
	return jiffies;
}

/*
 * This needs some heavy checking ...
 * I just haven't the stomach for it. I also don't fully
 * understand sessions/pgrp etc. Let somebody who does explain it.
 *
 * OK, I think I have the protection semantics right.... this is really
 * only important on a multi-user system anyway, to make sure one user
 * can't send a signal to a process owned by another.  -TYT, 12/12/91
 *
 * Auch. Had to add the 'did_exec' flag to conform completely to POSIX.
 * LBT 04.03.94
 */

asmlinkage int sys_setpgid(pid_t pid, pid_t pgid)
{
	struct task_struct * p;
	int err = -EINVAL;

	if (!pid)
		pid = current->pid;
	if (!pgid)
		pgid = pid;
	if (pgid < 0)
		return -EINVAL;

	/* From this point forward we keep holding onto the tasklist lock
	 * so that our parent does not change from under us. -DaveM
	 */
	read_lock(&tasklist_lock);

	err = -ESRCH;
	p = find_task_by_pid(pid);
	if (!p)
		goto out;

	if (p->p_pptr == current || p->p_opptr == current) {
		err = -EPERM;
		if (p->session != current->session)
			goto out;
		err = -EACCES;
		if (p->did_exec)
			goto out;
	} else if (p != current)
		goto out;
	err = -EPERM;
	if (p->leader)
		goto out;
	if (pgid != pid) {
		struct task_struct * tmp;
		for_each_task (tmp) {
			if (tmp->pgrp == pgid &&
			    tmp->session == current->session)
				goto ok_pgid;
		}
		goto out;
	}

ok_pgid:
	p->pgrp = pgid;
	err = 0;
out:
	/* All paths lead to here, thus we are safe. -DaveM */
	read_unlock(&tasklist_lock);
	return err;
}

asmlinkage int sys_getpgid(pid_t pid)
{
	if (!pid) {
		return current->pgrp;
	} else {
		int retval;
		struct task_struct *p;

		read_lock(&tasklist_lock);
		p = find_task_by_pid(pid);

		retval = -ESRCH;
		if (p)
			retval = p->pgrp;
		read_unlock(&tasklist_lock);
		return retval;
	}
}

asmlinkage int sys_getpgrp(void)
{
	/* SMP - assuming writes are word atomic this is fine */
	return current->pgrp;
}

asmlinkage int sys_getsid(pid_t pid)
{
	if (!pid) {
		return current->session;
	} else {
		int retval;
		struct task_struct *p;

		read_lock(&tasklist_lock);
		p = find_task_by_pid(pid);

		retval = -ESRCH;
		if(p)
			retval = p->session;
		read_unlock(&tasklist_lock);
		return retval;
	}
}

asmlinkage int sys_setsid(void)
{
	struct task_struct * p;
	int err = -EPERM;

	read_lock(&tasklist_lock);
	for_each_task(p) {
		if (p->pgrp == current->pid)
			goto out;
	}

	current->leader = 1;
	current->session = current->pgrp = current->pid;
	current->tty = NULL;
	current->tty_old_pgrp = 0;
	err = current->pgrp;
out:
	read_unlock(&tasklist_lock);
	return err;
}

/*
 * Supplementary group IDs
 */
asmlinkage int sys_getgroups(int gidsetsize, gid_t *grouplist)
{
	int i;
	
	/*
	 *	SMP: Nobody else can change our grouplist. Thus we are
	 *	safe.
	 */

	if (gidsetsize < 0)
		return -EINVAL;
	i = current->ngroups;
	if (gidsetsize) {
		if (i > gidsetsize)
			return -EINVAL;
		if (copy_to_user(grouplist, current->groups, sizeof(gid_t)*i))
			return -EFAULT;
	}
	return i;
}

/*
 *	SMP: Our groups are not shared. We can copy to/from them safely
 *	without another task interfering.
 */
 
asmlinkage int sys_setgroups(int gidsetsize, gid_t *grouplist)
{
	if (!capable(CAP_SETGID))
		return -EPERM;
	if ((unsigned) gidsetsize > NGROUPS)
		return -EINVAL;
	if(copy_from_user(current->groups, grouplist, gidsetsize * sizeof(gid_t)))
		return -EFAULT;
	current->ngroups = gidsetsize;
	return 0;
}

int in_group_p(gid_t grp)
{
	if (grp != current->fsgid) {
		int i = current->ngroups;
		if (i) {
			gid_t *groups = current->groups;
			do {
				if (*groups == grp)
					goto out;
				groups++;
				i--;
			} while (i);
		}
		return 0;
	}
out:
	return 1;
}

/*
 * This should really be a blocking read-write lock
 * rather than a semaphore. Anybody want to implement
 * one?
 */
struct semaphore uts_sem = MUTEX;

asmlinkage int sys_newuname(struct new_utsname * name)
{
	int errno = 0;

	down(&uts_sem);
	if (copy_to_user(name,&system_utsname,sizeof *name))
		errno = -EFAULT;
	up(&uts_sem);
	return errno;
}

asmlinkage int sys_sethostname(char *name, int len)
{
	int errno;

	if (!capable(CAP_SYS_ADMIN))
		return -EPERM;
	if (len < 0 || len > __NEW_UTS_LEN)
		return -EINVAL;
	down(&uts_sem);
	errno = -EFAULT;
	if (!copy_from_user(system_utsname.nodename, name, len)) {
		system_utsname.nodename[len] = 0;
		errno = 0;
	}
	up(&uts_sem);
	return errno;
}

asmlinkage int sys_gethostname(char *name, int len)
{
	int i, errno;

	if (len < 0)
		return -EINVAL;
	down(&uts_sem);
	i = 1 + strlen(system_utsname.nodename);
	if (i > len)
		i = len;
	errno = 0;
	if (copy_to_user(name, system_utsname.nodename, i))
		errno = -EFAULT;
	up(&uts_sem);
	return errno;
}

/*
 * Only setdomainname; getdomainname can be implemented by calling
 * uname()
 */
asmlinkage int sys_setdomainname(char *name, int len)
{
	int errno;

	if (!capable(CAP_SYS_ADMIN))
		return -EPERM;
	if (len < 0 || len > __NEW_UTS_LEN)
		return -EINVAL;

	down(&uts_sem);
	errno = -EFAULT;
	if (!copy_from_user(system_utsname.domainname, name, len)) {
		errno = 0;
		system_utsname.domainname[len] = 0;
	}
	up(&uts_sem);
	return errno;
}

asmlinkage int sys_getrlimit(unsigned int resource, struct rlimit *rlim)
{
	if (resource >= RLIM_NLIMITS)
		return -EINVAL;
	else
		return copy_to_user(rlim, current->rlim + resource, sizeof(*rlim))
			? -EFAULT : 0;
}

asmlinkage int sys_setrlimit(unsigned int resource, struct rlimit *rlim)
{
	struct rlimit new_rlim, *old_rlim;

	if (resource >= RLIM_NLIMITS)
		return -EINVAL;
	if(copy_from_user(&new_rlim, rlim, sizeof(*rlim)))
		return -EFAULT;
	if (new_rlim.rlim_cur < 0 || new_rlim.rlim_max < 0)
		return -EINVAL;
	old_rlim = current->rlim + resource;
	if (((new_rlim.rlim_cur > old_rlim->rlim_max) ||
	     (new_rlim.rlim_max > old_rlim->rlim_max)) &&
	    !capable(CAP_SYS_RESOURCE))
		return -EPERM;
	if (resource == RLIMIT_NOFILE) {
		if (new_rlim.rlim_cur > NR_OPEN || new_rlim.rlim_max > NR_OPEN)
			return -EPERM;
	}
	*old_rlim = new_rlim;
	return 0;
}

/*
 * It would make sense to put struct rusage in the task_struct,
 * except that would make the task_struct be *really big*.  After
 * task_struct gets moved into malloc'ed memory, it would
 * make sense to do this.  It will make moving the rest of the information
 * a lot simpler!  (Which we're not doing right now because we're not
 * measuring them yet).
 *
 * This is SMP safe.  Either we are called from sys_getrusage on ourselves
 * below (we know we aren't going to exit/disappear and only we change our
 * rusage counters), or we are called from wait4() on a process which is
 * either stopped or zombied.  In the zombied case the task won't get
 * reaped till shortly after the call to getrusage(), in both cases the
 * task being examined is in a frozen state so the counters won't change.
 */
int getrusage(struct task_struct *p, int who, struct rusage *ru)
{
	struct rusage r;

	memset((char *) &r, 0, sizeof(r));
	switch (who) {
		case RUSAGE_SELF:
			r.ru_utime.tv_sec = CT_TO_SECS(p->times.tms_utime);
			r.ru_utime.tv_usec = CT_TO_USECS(p->times.tms_utime);
			r.ru_stime.tv_sec = CT_TO_SECS(p->times.tms_stime);
			r.ru_stime.tv_usec = CT_TO_USECS(p->times.tms_stime);
			r.ru_minflt = p->min_flt;
			r.ru_majflt = p->maj_flt;
			r.ru_nswap = p->nswap;
			break;
		case RUSAGE_CHILDREN:
			r.ru_utime.tv_sec = CT_TO_SECS(p->times.tms_cutime);
			r.ru_utime.tv_usec = CT_TO_USECS(p->times.tms_cutime);
			r.ru_stime.tv_sec = CT_TO_SECS(p->times.tms_cstime);
			r.ru_stime.tv_usec = CT_TO_USECS(p->times.tms_cstime);
			r.ru_minflt = p->cmin_flt;
			r.ru_majflt = p->cmaj_flt;
			r.ru_nswap = p->cnswap;
			break;
		default:
			r.ru_utime.tv_sec = CT_TO_SECS(p->times.tms_utime + p->times.tms_cutime);
			r.ru_utime.tv_usec = CT_TO_USECS(p->times.tms_utime + p->times.tms_cutime);
			r.ru_stime.tv_sec = CT_TO_SECS(p->times.tms_stime + p->times.tms_cstime);
			r.ru_stime.tv_usec = CT_TO_USECS(p->times.tms_stime + p->times.tms_cstime);
			r.ru_minflt = p->min_flt + p->cmin_flt;
			r.ru_majflt = p->maj_flt + p->cmaj_flt;
			r.ru_nswap = p->nswap + p->cnswap;
			break;
	}
	return copy_to_user(ru, &r, sizeof(r)) ? -EFAULT : 0;
}

asmlinkage int sys_getrusage(int who, struct rusage *ru)
{
	if (who != RUSAGE_SELF && who != RUSAGE_CHILDREN)
		return -EINVAL;
	return getrusage(current, who, ru);
}

asmlinkage int sys_umask(int mask)
{
	mask = xchg(&current->fs->umask, mask & S_IRWXUGO);
	return mask;
}
    
asmlinkage int sys_prctl(int option, unsigned long arg2, unsigned long arg3,
			 unsigned long arg4, unsigned long arg5)
{
	int error = 0;
	int sig;

	switch (option) {
		case PR_SET_PDEATHSIG:
			sig = arg2;
			if (sig > _NSIG) {
				error = -EINVAL;
				break;
			}
			current->pdeath_signal = sig;
			break;
		default:
			error = -EINVAL;
			break;
	}
	return error;
}

